/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef AtomDC_H
#define AtomDC_H

#include <set>
#include <Component.h>
#include <GeometricObject.h>

#include <vtkFollower.h>
#include <vtkSmartPointer.h>
class QPixmap;
class QMenu;

#include "PMComponentAPI.h"
class PMManagerDC;
class Atom;
class StructuralComponentDC;
class AtomDCProperties;
class AtomDCPopup;
class AtomDecoration;

namespace std {

    /** definition of a couple (=STL pair) [InterfaceNode *, unsigned int]
        * this associates a parent data component with a subItem order number
      */
    typedef std::pair<camitk::InterfaceNode *, unsigned int> IndexInParentItemPair;
    /** definition of the association set (=map in STL) IndexInParentItemMap.
      * IndexInParentItemMap allows a DC to associate all its parent DCs with
      * its order numbers in these parent DCs.
      * The key is the  parent DC, so that it is simple to retrieve the order number of
      * a this DC in a particular parent item.
      */
    typedef std::map <camitk::InterfaceNode *, unsigned int> IndexInParentItemMap;
    /** the iterator corresponding to the IndexInParentItemMap map */
    typedef std::map <camitk::InterfaceNode *, unsigned int>::iterator IndexInParentItemMapIterator;
}

/**The manager of the Atom data.
 * AtomDC are only created in the StructuralComponentDC constructor,
 * which garanties that each atom has ONE AND ONLY ONE DC.
 * But as Atom can be referenced by more than one Component (StructuralComponent or Cell)
 * AtomDC is a particular type of DC that can have more than one parent DC (setParent is
 * rewritten here).
 *
 * 
 */
class PHYSICALMODEL_COMPONENT_API AtomDC : public camitk::Component {
Q_OBJECT  
public:
    AtomDC(camitk::Component *parent, PMManagerDC * pmManagerDC, Atom *);
    virtual ~AtomDC();

    /// @name Redefined from camitk::Component
    ///@{
    /** An AtomDC can have more than one parent...
      * myParentDC is the first DC (atom list).
      */
    virtual void setParent(InterfaceNode *);

    virtual QPixmap getIcon();

    /// picking is selecting/unselecting
    void pointPicked(vtkIdType, bool);
    
    /// when seleceted add a sphere glyph around the atom position and set the enhanced mode to Normal
    virtual void setSelected(const bool, const bool recursive=false);

    /// do nothing here (the enhanced mode is controled by the setSelected method)
    virtual void setEnhancedModes(const EnhancedModes);

    /// set the name of the atom as well as the name of the DC
    virtual void setName(const QString &);

    /// if the vtkPointSet is modified externally (e.g. by an operator), change the PM structure
    virtual void setPointSet(vtkSmartPointer<vtkPointSet>);

    /// Overriden method so that we actually can build a popup menu with different actions
    virtual QMenu * getPopupMenu(QWidget* parent);

    /// reimplement this method to give the property widget to the app
    virtual QWidget * getPropertyWidget(QWidget* parent = 0);

    /// reimplement this method to give the property object to the property editor
    virtual QObject * getPropertyObject();

    /// overloaded method, allows to set the point when double clicked
    virtual bool doubleClicked();
    ///@}

    /// set a new position for this atom
    void setPosition(double, double, double);

    /// get the current position for this atom
    void getPosition(double &, double &, double &);

    /** update the position of the 3D representation and launch the update position cascade process
      * (updating all the 3D representation of all the SC/cells containing this atom, then of
      * all the SC containing all the cells containing this atom...)
      */
    void updatePosition();

    /// reset the alreadyMoved flag to false, see alreadyMoved property
    void resetAlreadyMovedFlag();

    /// get the atom this dc is representing
    Atom * getAtom();

    /// get the PMManagerDC (given during instanciation), allows to get PML/Component maps
    virtual PMManagerDC * getPMManagerDC() { return myPMManagerDC; }
    
    /** register a StructuralComponentDC and this AtomDC corresponding order number in it (NOTE: this
      * method replace the previous index if there was one)
      * @return the new number of registered SCDC
      */
    unsigned int registerIndexInSCDC(const std::IndexInParentItemPair);

    /** unregister a StructuralComponentDC item from the map
      * @return the new number of registered SCDC
      */
    unsigned int unregisterIndexInSCDC(StructuralComponentDC *);

    /// knowing a StructuralComponentDC get the order number of this AtomDC in it (return -1 if not found)
    int getOrderNumberInSCDC(StructuralComponentDC *);

    /// add a ptr to a pointData for this atom corresponding to the given SCDC
    void addPointData(StructuralComponentDC *, double *);

    /// clear the point data ptr list
    void clearPointData();

    /// update point data value
    void updatePointData(const double);

    /** get a decoration of the AtomDC using its name.
      * If the AtomDC has no decoration using this name, create a new one using the given type.
      *
      * @param name the decoration name
      * @param t the decoration type
      */
    AtomDecoration *getDecoration(const QString & name, camitk::GeometricObject::Geometry t);

  private:

    /// The atom the dc is representing
    Atom *myAtom;

    /// the concrete building of the geometry
    virtual void initRepresentation();
    
    /** true if the atom has been already moved by another SC
     *  This is a way to solve the bug which occurs when 2 selected
     *  cells has this atom in common. The call of setRealTransformation()
     *  being sequencial, the first cell will move the atom first, and
     *  then the second cell will move it again...
     *  Set to false when setTranform(...) is called with the identity.
     *  Set to true when updatePosition is called
     */
    bool alreadyMoved;

    /** correspondance between a SCDC and the order number of this atom dc in a
      * particular SCDC (parent items).
      */
    std::IndexInParentItemMap mySCDCindexes;

    /// the popup menu
    AtomDCPopup * myPopupMenu;

    /// the atom'properties
    AtomDCProperties * myProp;

    /// the AtomDC pixmap
    static QPixmap * myPixmap;

    /// list of all the point data adresses
    std::vector<double *> pointData;

    /// list of all the SC who asked for a point data in this AtomDC
    std::set<StructuralComponentDC *> pointDataSC;

    /// list of decorations attached to this AtomDC
    QMap<QString, AtomDecoration*> decorations;

    /// the PMManagerDC
    PMManagerDC *myPMManagerDC;

};

inline Atom * AtomDC::getAtom() {
  return myAtom;
}

#endif
