/*
 * Copyright 2002 Sun Microsystems, Inc.  All rights reserved.
 *
 * Use is subject to the GNU Lesser General Public License, Version 2.1,
 * February 1999, which is contained in the read-me file named
 * "README_GNU_LGPL." This library is free software; you can
 * redistribute it and/or modify it under the terms of the GNU
 * Lesser General Public License as published by the Free Software
 * Foundation; either version 2.1 of the License, or (at your
 * option) any later version. This library is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA.
 *
 * implement time-related functions for panel utils. it's done this way
 * because select() doesn't always work with the lcd driver. bleah.
 *
 * NOTE: as we want to make sure that all of the lcd utils know
 *       about each other, this is almost intentionally 
 *       single-threaded.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include "lcdutils.h"

static struct itimerval timeout;

/* start up an alarm handler for timeouts. we have to do it
 * this way because we can't do a select/poll on /dev/lcd
 * and have it be meaningful. */
int lcd_timeout_start(void (*fcn)(int), const int seconds) {
	struct sigaction act;
	
	memset(&act, 0, sizeof(act));
	act.sa_handler = fcn;
	act.sa_flags = SA_RESTART;
	if (sigaction(SIGALRM, &act, NULL) < 0)
		return -1;
	
	memset(&timeout, 0, sizeof(timeout));
	timeout.it_interval.tv_sec = seconds;
	timeout.it_value.tv_sec = seconds;
	if (setitimer(ITIMER_REAL, &timeout, NULL) < 0) {
		sigaction(SIGALRM, NULL, NULL);
		return -1;
	}
	
	return 0;
}

/* stop and reset everything */
void lcd_timeout_stop(void) {
	struct itimerval zero;
	struct sigaction act;
	
	memset(&zero, 0, sizeof(zero));
	memset(&act, 0, sizeof(act));
	setitimer(ITIMER_REAL, &zero, NULL);
	act.sa_handler = SIG_DFL;
	sigaction(SIGALRM, &act, NULL);
}

/* just do a reset */
int lcd_timeout_reset(void) {
	return setitimer(ITIMER_REAL, &timeout, NULL); 
}
