/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using Gtk;
using Glade;
using Cowbell.Base;

namespace Cowbell.Gui
{
	public class MainWindow : GladeWindow
	{
		/* public properties */
		public Window Window {
			get { return window; }
		}

		/* public methods */
		public MainWindow () : base ("MainWindow.glade", "window")
		{
			busy_cursor = new Gdk.Cursor (Gdk.CursorType.Watch);
			
			AppDomain.CurrentDomain.UnhandledException += new UnhandledExceptionEventHandler (OnUnhandledException);

			Runtime.Database.Modified += new DatabaseModifiedHandler (OnDatabaseModified);
			Runtime.Database.GlobalDataModified += new VoidHandler (OnGlobalDataModified);
			Runtime.Database.MultipleArtistsChanged += new VoidHandler (OnMultipleArtistsChanged);
			Runtime.Database.MultipleDiscsChanged += new VoidHandler (OnMultipleDiscsChanged);

			window.DeleteEvent += new DeleteEventHandler (OnMainWindowDeleteEvent);
			ArtistEntry.Changed += new EventHandler (OnEntryChanged);
			AlbumEntry.Changed += new EventHandler (OnEntryChanged);
			YearEntry.Changed += new EventHandler (OnEntryChanged);
			GenreComboBox.Changed += new EventHandler (OnEntryChanged);
			MultipleArtistCheckBox.Toggled += new EventHandler (OnMultipleArtistCheckBoxToggled);
			MultipleDiscCheckBox.Toggled += new EventHandler (OnMultipleDiscCheckBoxToggled);

			ArtistEntry.FocusOutEvent += new FocusOutEventHandler (OnEntryFocusOutEvent);
			AlbumEntry.FocusOutEvent += new FocusOutEventHandler (OnEntryFocusOutEvent);
			YearEntry.FocusOutEvent += new FocusOutEventHandler (OnEntryFocusOutEvent);
			GenreComboBox.FocusOutEvent += new FocusOutEventHandler (OnEntryFocusOutEvent);

			about.Activated += new EventHandler (OnHelpAboutActivate);
			open.Activated += new EventHandler (OnFileOpenActivate);
			save.Activated += new EventHandler (OnFileSaveActivate);
			revert.Activated += new EventHandler (OnFileRevertActivate);
			quit.Activated += new EventHandler (OnFileQuitActivate);
			guess_song_information.Activated += new EventHandler (OnGuessSongInformationActivate);
			preferences.Activated += new EventHandler (OnEditPreferencesActivate);
			clear_list.Activated += new EventHandler (OnEditClearListActivate);

			GLib.Timeout.Add (3000, new GLib.TimeoutHandler (OnCoverImageUpdateTimeout));

			window.ShowAll ();
			Freeze ();
		}

		public void SaveCoverImage (string filename)
		{
			cover.Save (filename);
		}

		/* private fields */
		[Glade.Widget] Window window;
		[Glade.Widget] AlbumCoverImage cover;
		[Glade.Widget] MenuItem open, save, revert, quit, guess_song_information, preferences, clear_list, about;
		[Glade.Widget] ToggleButton MultipleArtistCheckBox, MultipleDiscCheckBox;
		[Glade.Widget] Entry ArtistEntry, AlbumEntry, YearEntry;
		[Glade.Widget] ComboBoxEntry GenreComboBox;
		[Glade.Widget] Label AlbumLabel, PlayTimeLabel, TracksLabel;
		[Glade.Widget] HPaned MainHPaned;

		private Gdk.Cursor busy_cursor;

		/* private methods */
		private void Freeze ()
		{
			save.Sensitive = false;
			revert.Sensitive = false;
			clear_list.Sensitive = false;
			guess_song_information.Sensitive = false;
			ArtistEntry.Sensitive = false;
			AlbumEntry.Sensitive = false;
			YearEntry.Sensitive = false;
			GenreComboBox.Sensitive = false;	
			MultipleArtistCheckBox.Sensitive = false;
			MultipleDiscCheckBox.Sensitive = false;
		}

		private void Thaw ()
		{
			revert.Sensitive = true;
			clear_list.Sensitive = true;
			guess_song_information.Sensitive = true;
			YearEntry.Sensitive = true;
			GenreComboBox.Sensitive = true;	
			MultipleArtistCheckBox.Sensitive = true;
			MultipleDiscCheckBox.Sensitive = true;
		}

		private bool ConfirmQuit ()
		{
			if (!Runtime.Dirty)
				return true;

			SaveChangesDialog d = new SaveChangesDialog (Runtime.Database.GlobalData.Album);

			ResponseType resp = (ResponseType)d.Run ();
			d.Hide ();

			switch (resp) {
			case ResponseType.Yes:
				Filesystem.Save ();
				break;

			case ResponseType.No:
				break;

			case ResponseType.Cancel:
			case ResponseType.DeleteEvent:
			default:
				return false;
			}

			return true;
		}

		private void OnDatabaseModified (int location, ModificationType type)
		{
			Thaw ();

			if (type == ModificationType.Change) {
				save.Sensitive = true;
				Runtime.Dirty = true;
				return;
			}

			// update the dynamic data fields like Play Time and Tracks
			PlayTimeLabel.Text = String.Format (Catalog.GetString ("Play Time: {0}"),
			                                    Utils.HumanReadableTimeSpan (Runtime.Database.PlayTime));

			TracksLabel.Text = String.Format (Catalog.GetString ("Tracks: {0}"),
			                                  Runtime.Database.Count);

			// check for new album art
			cover.CheckFilesystemForAlbumArt (Filesystem.GetBasedir ());

			// XXX: Hack around a stupid gtk bug
			MainHPaned.QueueDraw ();
		}

		private void OnGlobalDataModified ()
		{
			AlbumLabel.Markup = String.Format ("<span weight='bold' size='x-large'>{0}</span>",
			                                   Runtime.Database.GlobalData.Album == String.Empty ? Catalog.GetString ("No Album") : GLib.Markup.EscapeText (Runtime.Database.GlobalData.Album));

			/*
			 * XXX: Shoot me, please
			 */
			ArtistEntry.Changed -= new EventHandler (OnEntryChanged);
			ArtistEntry.Text = Runtime.Database.GlobalData.Artist;
			ArtistEntry.Changed += new EventHandler (OnEntryChanged);

			AlbumEntry.Changed -= new EventHandler (OnEntryChanged);
			AlbumEntry.Text = Runtime.Database.GlobalData.Album;
			AlbumEntry.Changed += new EventHandler (OnEntryChanged);

			YearEntry.Changed -= new EventHandler (OnEntryChanged);
			YearEntry.Text = Convert.ToString (Runtime.Database.GlobalData.Year);
			YearEntry.Changed += new EventHandler (OnEntryChanged);

			GenreComboBox.Changed -= new EventHandler (OnEntryChanged);
			((Entry)GenreComboBox.Child).Text = Runtime.Database.GlobalData.Genre;
			GenreComboBox.Changed += new EventHandler (OnEntryChanged);

			ArtistEntry.Sensitive = !Runtime.Database.MultipleArtists;
			AlbumEntry.Sensitive = !Runtime.Database.MultipleDiscs;
		}

		private void OnFileQuitActivate (object o, EventArgs args)
		{
			if (ConfirmQuit ()) {
				Runtime.Preferences.Save ();
				Application.Quit ();
			}
		}

		private void OnMainWindowDeleteEvent (object o, DeleteEventArgs args)
		{
			args.RetVal = true;
			if (ConfirmQuit ()) {
				Runtime.Preferences.Save ();
				Application.Quit ();
				args.RetVal = false;
			}
		}

		private void OnFileOpenActivate (object o, EventArgs args)
		{
			FileChooser chooser = new FileChooser ();
			ResponseType response;
	
			response = (ResponseType)chooser.Run ();
			chooser.Hide ();

			if (response == ResponseType.Accept) {
				Filesystem.Import (chooser.Uris);
			}

			if (Runtime.Database.Count > 0) {
				Thaw ();
			}

			if (Runtime.Database.Count > 25) {
				/* NOTE: This isn't Catalog.GetPluralString
				 * because we're guaranteed to have more than
				 * 25 songs.  If this is a problem in your
				 * language, let me know. */
				Runtime.Gui.ShowError (String.Format (Catalog.GetString ("You have just imported {0} songs, generally more than one album worth of music.\n\nCowbell is intended for tagging just one album at a time, and will not work effectively if given more than one album.  To tag multiple albums at a time, Cowbell has a batch tagging mode, enabled by passing \"--batch\" on the console.\n\nIf you are only trying to tag one album, you can safely ignore this message."), Runtime.Database.Count));
			}

			cover.Clear ();
			OnCoverImageUpdateTimeout ();
		}

		private void OnFileSaveActivate (object o, EventArgs args)
		{
			MainHPaned.Sensitive = false;

			window.GdkWindow.Cursor = busy_cursor;
			
			Runtime.Dispatcher.BackgroundDispatch (new VoidHandler (ThreadedSave));
		}

		private void ThreadedSave ()
		{
			Filesystem.Save ();

			Runtime.Dispatcher.GuiDispatch (new VoidHandler (ThreadedSaveCleanup));
		}

		private void ThreadedSaveCleanup ()
		{
			MainHPaned.Sensitive = true;
			Runtime.Dirty = false;

			window.GdkWindow.Cursor = null;
		}
		
		private void OnFileRevertActivate (object o, EventArgs args)
		{
			Runtime.Database.RevertAll ();
			Runtime.Dirty = false;
		}

		private void OnGuessSongInformationActivate (object o, EventArgs args)
		{
			MetadataProxy p = new MetadataProxy ();
			p.Import ();
		}

		private void OnEditPreferencesActivate (object o, EventArgs args)
		{
			PreferencesWindow pw = new PreferencesWindow ();
			pw.Show ();
			
			/* just sensitize the save icon, but don't set the app
			 * dirty (because this would make you have to save
			 * before you exit) */
			save.Sensitive = true;
		}

		private void OnEditClearListActivate (object o, EventArgs args)
		{
			Runtime.Database.Clear ();
			Runtime.Database.GlobalData = new Song ();
			cover.Clear ();
			Runtime.Dirty = false;

			Runtime.Database.MultipleArtists = false;
			Runtime.Database.MultipleDiscs = false;
		}

		private void OnHelpAboutActivate (object o, EventArgs args)
		{
			AboutDialog about = new AboutDialog ();
			about.Run ();
			about.Hide ();
		}

		private void OnEntryChanged (object o, EventArgs args)
		{
			Entry e = o as Entry;
			ComboBoxEntry c = o as ComboBoxEntry;

			switch (((Widget)o).Name) {
			case "ArtistEntry":
				Runtime.Database.GlobalData.Artist = e.Text;
				break;

			case "AlbumEntry":
				Runtime.Database.GlobalData.Album = e.Text;
				break;

			case "YearEntry":
				try {
					Runtime.Database.GlobalData.Year = Convert.ToUInt32 (e.Text);
				} catch {
					Runtime.Database.GlobalData.Year = 0;
				}
				break;

			case "GenreComboBox":
				Runtime.Database.GlobalData.Genre = ((Entry)c.Child).Text;
				break;
			}

			Runtime.Dirty = true;
			save.Sensitive = true;
		}

		private void OnEntryFocusOutEvent (object o, FocusOutEventArgs args)
		{
			OnCoverImageUpdateTimeout ();
		}

		private void OnMultipleArtistCheckBoxToggled (object o, EventArgs args)
		{
			Runtime.Database.MultipleArtists = ((ToggleButton)o).Active;
		}

		private void OnMultipleDiscCheckBoxToggled (object o, EventArgs args)
		{
			Runtime.Database.MultipleDiscs = ((ToggleButton)o).Active;
		}

		private void OnMultipleArtistsChanged ()
		{
			OnCoverImageUpdateTimeout ();
			ArtistEntry.Sensitive = !Runtime.Database.MultipleArtists;
			
			/* Prevent circular eventing badness */
			MultipleArtistCheckBox.Toggled -= new EventHandler (OnMultipleArtistCheckBoxToggled);
			MultipleArtistCheckBox.Active = Runtime.Database.MultipleArtists;
			MultipleArtistCheckBox.Toggled += new EventHandler (OnMultipleArtistCheckBoxToggled);
		}
		
		private void OnMultipleDiscsChanged ()
		{
			AlbumEntry.Sensitive = !Runtime.Database.MultipleDiscs;
			
			/* Prevent circular eventing badness */
			MultipleDiscCheckBox.Toggled -= new EventHandler (OnMultipleDiscCheckBoxToggled);
			MultipleDiscCheckBox.Active = Runtime.Database.MultipleDiscs;
			MultipleDiscCheckBox.Toggled += new EventHandler (OnMultipleDiscCheckBoxToggled);
		}

		private bool OnCoverImageUpdateTimeout ()
		{
			cover.Download ();
			return true;
		}

		private void OnUnhandledException (object o, UnhandledExceptionEventArgs args)
		{
			UnhandledExceptionDialog d = new UnhandledExceptionDialog ((Exception)args.ExceptionObject);
			d.Run ();
			d.Hide ();
			Application.Quit ();
		}
	}
}
