#!/bin/bash

EASY_VERSION="1.1"

bb=`tput bold`
nn=`tput sgr0`

# help
if [ $# -eq 0 ]; then
   echo "${bb}-------------------------------------------------------------------${nn}"
   echo "${bb}Creates sof-file for kmo_multi_reconstruct-recipe and executes it.${nn}"
   echo " "
   echo "- The environment variable ${bb}KMOS_CALIB${nn} pointing to a path has to be"
   echo "  set in order to find the needed static calibration frame WAVE_BAND."
   echo "  Otherwise the execution of this script will fail!"
   echo "- If XCAL, YCAL, LCAL, MASTER_DARK and MASTER_FLAT frames aren't existing"
   echo "  in the working directory the script is searching as well KMOS_CALIB in"
   echo "  order to find them."
   echo "- As argument an arbitrary ${bb}KMOS_SPEC_OBS*${nn} exposure of an executed "
   echo "  template must be provided."
   echo "  All matching exposures are collected automatically."
   echo "- (Optional) If as 2nd argument ${bb}sof${nn} is provided, then only"
   echo "  the sof-file is generated but not executed."
   echo " "
   echo "${bb}Usage:${nn}"
   echo "  easySPARK_multi_reconstruct.sh <filename with full path>"
   echo "    or"
   echo "  easySPARK_multi_reconstruct.sh <filename with full path> sof"
   echo " "
   echo "Version: "$EASY_VERSION
   echo "${bb}-------------------------------------------------------------------${nn}"
   exit
fi

# init
sofOnly=0
printVers=0

# extract all arguments beginning with KMOS, eventually imethod and sof
while [ "$1" ] ; do
    tmp=$(basename $1)
    if [[ $tmp == KMOS* ]]; then
        # file-check
        if [ ! -e $1 ]; then
            echo "${bb}ERROR:${nn} File doesn't exist! ($1)"
            exit
        fi
        fileName=$1
    else
        if [[ $tmp == sof ]]; then
            sofOnly=1
        elif [[ $tmp == vers ]]; then
            printVers=1
        fi
    fi
    shift
done

if [ $printVers = 1 ]; then
    echo $EASY_VERSION
    exit
fi

# check if gawk is existing
if command -v gawk >/dev/null; then
   :
else
    echo "'gawk' has not been found on your system!"
    OS=$(uname -s)
    if [ $OS != "Linux" ]; then
        echo "You seem using a MacOS, please use e.g. MacPorts to install gawk."
    fi
    echo "Aborting now.!"
fi

if [ -z "${KMOS_CALIB}" ]; then
   echo "${bb}ERROR:${nn} environment variable KMOS_CALIB is not defined! (see help)"
   exit
fi

#check if file is of correct template
tplId=$(dfits $fileName |\
        fitsort -d tpl.id |\
        gawk '{print $2}')
if [[ $tplId != KMOS_spec_obs_nodtosky &&  $tplId != KMOS_spec_obs_stare &&  $tplId != KMOS_spec_obs_mapping8 &&  $tplId != KMOS_spec_obs_mapping24 &&  $tplId != KMOS_spec_obs_freedither ]]; then
   echo "${bb}ERROR${nn}: File is of wrong template: $tplId instead of KMOS_spec_obs_nodtosky, KMOS_spec_obs_stare, KMOS_spec_obs_mapping8, KMOS_spec_obs_mapping24 or KMOS_spec_obs_freedither !"
   exit
fi

# extract TPL.START of ths file
tplStartTime=$(dfits $fileName |\
               fitsort -d tpl.start |\
               gawk '{print $2}')

# get files with matching tplStartTime and OBJECT,SKY
dataDir=$(dirname "$fileName")
matchingFiles=$(dfits $dataDir/KMOS*.fits | \
                fitsort -d tpl.start dpr.type |\
       	        grep $tplStartTime |\
                grep "OBJECT,SKY" |\
                gawk '{print $1}' |\
                tr "\n" " ")

# populate sof-file with SCIENCE frames
sofFile="multi_reconstruct_$tplStartTime.sof"
cat > $sofFile </dev/null
for file in $matchingFiles; do
    echo "$file                SCIENCE" >> $sofFile
done

# get grating id of first SCIENCE frame
gratingUpper=$(dfits $matchingFiles |\
               fitsort -d ins.grat1.id |\
               head -n1 |\
               gawk '{print $2}')
gratingLower=$(echo $gratingUpper|tr "A-Z" "a-z")

# add static calibration frames
echo "$KMOS_CALIB/kmos_wave_band.fits       WAVE_BAND" >> $sofFile

# add band-specific calibration frames
fnMasterFlat=master_flat_$gratingUpper$gratingUpper$gratingUpper.fits
fnXcal=xcal_$gratingUpper$gratingUpper$gratingUpper.fits
fnYcal=ycal_$gratingUpper$gratingUpper$gratingUpper.fits
fnLcal=lcal_$gratingUpper$gratingUpper$gratingUpper.fits
fnIllum=illum_corr_$gratingUpper$gratingUpper$gratingUpper.fits
fnTell=telluric_$gratingUpper$gratingUpper$gratingUpper.fits

if [ ! -e $fnMasterDark ]; then
   fnMasterDark=$KMOS_CALIB"/"$fnMasterDark
fi
if [ ! -e $fnMasterFlat ]; then
   fnMasterFlat=$KMOS_CALIB"/"$fnMasterFlat
fi
if [ ! -e $fnXcal ]; then
   fnXcal=$KMOS_CALIB"/"$fnXcal
fi
if [ ! -e $fnYcal ]; then
   fnYcal=$KMOS_CALIB"/"$fnYcal
fi
if [ ! -e $fnLcal ]; then
   fnLcal=$KMOS_CALIB"/"$fnLcal
fi

# these are optional
if [ ! -e $fnIllum ]; then
   fnIllum=$KMOS_CALIB"/"$fnIllum
   if [ ! -e $fnIllum ]; then
      fnIllum=
   fi
fi
if [ ! -e $fnTell ]; then
   fnTell=$KMOS_CALIB"/"$fnTell
   if [ ! -e $fnTell ]; then
      fnTell=
   fi
fi

echo "$fnMasterFlat      MASTER_FLAT" >> $sofFile
echo "$fnXcal            XCAL" >> $sofFile
echo "$fnYcal            YCAL" >> $sofFile
echo "$fnLcal            LCAL" >> $sofFile
echo "$KMOS_CALIB/kmos_oh_spec_$gratingLower.fits    OH_SPEC" >> $sofFile
if [ ! -z $fnIllum ]; then
    echo "$fnIllum           ILLUM_CORR" >> $sofFile
fi
if [ ! -z $fnTell ]; then
    echo "$fnTell            TELLURIC" >> $sofFile
fi

echo ""
echo "${bb}------------------ easySPARK_multi_reconstruct ---------------------${nn}"
echo "${bb}INPUT${nn}"
echo "filename:  "$fileName
echo "with "
echo "TPL.START: "$tplStartTime
echo "GRATING:   "$gratingUpper
echo "EXPTIME:   "$exptime
echo ""
echo "${bb}OUTPUT${nn}"
echo $sofFile" with:"
cat $sofFile
echo ""

# execute kmo_multi_reconstruct
if [ $sofOnly = 0 ]; then
   echo "${bb}NOW EXECUTING SCIENCE REDUCTION:${nn}"
   echo "esorex kmo_multi_reconstruct "$sofFile
   
   esorex kmo_multi_reconstruct $sofFile
fi
echo "${bb}----------------------------------------------------------${nn}"
