/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ffgtk.h
 * \brief Main ffgtk header file
 */

#ifndef FFGTK_H
#define FFGTK_H

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <fcntl.h>
#include <math.h>
#include <pthread.h>
#include <unistd.h>
#include <libintl.h>
#include <locale.h>
#include <dirent.h>
#include <errno.h>
#include <assert.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdint.h>

#include <glib.h>
#include <glib/gprintf.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libxml/parser.h>

#include <strings.h>
#include <server.h>
#include <addressbook.h>
#include <xml.h>
#include <profiles.h>
#include <plugins.h>
#include <urlHandler.h>
#include <router.h>
#include <actions.h>
#include <dtmf.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define _(text)	gettext(text)

#if __APPLE__ && __MACH__
#define __MACOSX__
#endif

#include <osdep.h>

#define GTK_FLUSH	while ( gtk_events_pending() ) { gtk_main_iteration(); }

enum eDebugLevel {
	KERN_NONE,
	KERN_INFO,
	KERN_WARNING,
	KERN_DEBUG
};

typedef enum {
	CALL_TYPE_INCOMING = 0x01,
	CALL_TYPE_OUTGOING = 0x02,
	CALL_TYPE_CONNECT = 0x04,
	CALL_TYPE_DISCONNECT = 0x08
} eCallType;

#define PORT_ANALOG1	0x0001
#define PORT_ANALOG2	0x0002
#define PORT_ANALOG3	0x0004
#define PORT_ISDNALL	0x0008
#define PORT_ISDN1		0x0010
#define PORT_ISDN2		0x0020
#define PORT_ISDN3		0x0040
#define PORT_ISDN4		0x0080
#define PORT_ISDN5		0x0100
#define PORT_ISDN6		0x0200
#define PORT_ISDN7		0x0400
#define PORT_ISDN8		0x0800
#define PORT_DECT1		0x1000
#define PORT_DECT2		0x2000
#define PORT_DECT3		0x4000
#define PORT_DECT4		0x8000
#define PORT_DECT5		0x10000
#define PORT_DECT6		0x20000

#define PORT_IP1		0x40000
#define PORT_IP2		0x80000
#define PORT_IP3		0x100000
#define PORT_IP4		0x200000
#define PORT_IP5		0x400000
#define PORT_IP6		0x800000
#define PORT_IP7		0x1000000
#define PORT_IP8		0x2000000
#define PORT_IP9		0x4000000
#define PORT_IP10		0x8000000

#define PORT_SOFTPHONE	0x10000000

#define BUF_SIZE	1024

void Debug2( int nLevel, const char *pnName, const char *pnFormat, ... );
#define Debug( __nLevel, format... )\
	do {\
		Debug2( __nLevel, __FUNCTION__, format );\
	} while ( 0 )


#define EXPORT_CSV_FRITZBOX	"Typ;Datum;Rufnummer;Nebenstelle;Eigene Rufnummer;Dauer"
#define EXPORT_CSV_FRITZBOX_NEWFIRMWARE	"Typ;Datum;Name;Rufnummer;Nebenstelle;Eigene Rufnummer;Dauer"
#define EXPORT_CSV_FRITZBOX_NEWFIRMWARE_EN	"Type;Date;Name;Number;Extension;Outgoing Caller ID;Duration"
#define EXPORT_CSV_THUNDERBIRD	"Vorname,Nachname,Anzeigename,Spitzname,Primäre E-Mail,Sekundäre E-Mail,Tel. dienstlich,Tel. privat,Fax-Nummer,Pager-Nummer,Mobil-Tel.-Nr.,Privat: Adresse,Privat: Adresse 2,Privat: Stadt,Privat: Bundesland,Privat: PLZ,Privat: Land,Dienstlich: Adresse,Dienstlich: Adresse 2,Dienstlich: Stadt,Dienstlich: Bundesland,Dienstlich: PLZ,Dienstlich: Land,Arbeitstitel,Abteilung,Organisation,Webseite 1,Webseite 2,Geburtsjahr,Geburtsmonat,Geburtstag,Benutzerdef. 1,Benutzerdef. 2,Benutzerdef. 3,Benutzerdef. 4,Notizen,"
#define EXPORT_CSV_FFGTK	"Name;Work;Home;Fax;Pager;Mobile"
#define EXPORT_CSV_FFGTK2	"Title;FirstName;LastName;DisplayName;Company;PrivatePhone;PrivateFax;PrivateMobile;PrivateStreet;PrivateZipCode;PrivateCity;PrivateCountry;BusinessPhone;BusinessFax;BusinessStreet;BusinessZipCode;BusinessCity;BusinessCountry"
#define EXPORT_CSV_SPEEDPORT	"Datum;Telefonnummer;Nebenstelle;Eigene Telefonnummer;Dauer"
#define IMPORT_CSV_JFRITZ	"\"Private\";\"Last Name\";\"First Name\";\"Company\";\"Street\";\"ZIP Code\";\"City\";\"E-Mail\";\"Picture\";\"Home\";\"Mobile\";\"Homezone\";\"Business\";\"Other\";\"Fax\";\"Sip\";\"Main\""
#define AREACODES "Vorwahl;Ortsnetz"

#define PERSON_ID				0x00001
#define PERSON_FIRST_NAME		0x00002
#define PERSON_LAST_NAME		0x00004
#define PERSON_DISPLAY_NAME		0x00008
#define PERSON_COMPANY			0x00010
#define PERSON_TITLE			0x00020
#define PERSON_IMAGE			0x00040
#define PERSON_PRIVATE_PHONE	0x00080
#define PERSON_PRIVATE_MOBILE   0x00100
#define PERSON_PRIVATE_FAX		0x00200
#define PERSON_PRIVATE_STREET   0x00400
#define PERSON_PRIVATE_CITY		0x00800
#define PERSON_PRIVATE_ZIPCODE  0x01000
#define PERSON_PRIVATE_COUNTRY  0x02000
#define PERSON_BUSINESS_PHONE   0x04000
#define PERSON_BUSINESS_FAX		0x08000
#define PERSON_BUSINESS_STREET  0x10000
#define PERSON_BUSINESS_CITY	0x20000
#define PERSON_BUSINESS_ZIPCODE 0x40000
#define PERSON_BUSINESS_COUNTRY 0x80000
#define PERSON_IMAGE_TYPE		0x100000
#define PERSON_CATEGORY			0x200000

#define PERSON_FLAG_UNCHANGED	0x0000
#define PERSON_FLAG_CHANGED		0x0001
#define PERSON_FLAG_DELETED		0x0002
#define PERSON_FLAG_NEW			0x0004

struct sPerson {
	gint nFlags;
	gchar *pnId;
	gchar *pnTitle;
	gchar *pnFirstName;
	gchar *pnLastName;
	gchar *pnDisplayName;
	gchar *pnCompany;
	gchar *pnPrivatePhone;
	gchar *pnPrivateFax;
	gchar *pnPrivateMobile;
	gchar *pnPrivateStreet;
	gchar *pnPrivateZipCode;
	gchar *pnPrivateCity;
	gchar *pnPrivateCountry;
	gchar *pnBusinessPhone;
	gchar *pnBusinessFax;
	gchar *pnBusinessStreet;
	gchar *pnBusinessZipCode;
	gchar *pnBusinessCity;
	gchar *pnBusinessCountry;
	gchar *pnNewImage;
	gchar *pnImageType;
	gchar *pnCategory;
	GdkPixbuf *psImage;
};

struct sCaller {
	gint nType;
	gchar *pnDateTime;
	gchar *pnName;
	gchar *pnNumber;
	gchar *pnLocalName;
	gchar *pnLocalNumber;
	gchar *pnDuration;
	gchar *pnProfile;
};

struct sDiversity {
	gchar *pnName;
	gint nIndex;
	gboolean bSet;
};

/** external svn revision array */
extern const gchar anSvnRevision[];

/* about-dialog.c */
void aboutDialog( void );

/* address-add.c */
gint AddAddressDialog( struct sPerson *psPerson, const gchar *pnNumber, gint nState );

/* actions.c */
void ActionsLoad( void );
void saveActions( void );
void executeActions( gint nCallType, gint nCurrentType, gchar *pnLocal, gchar *pnNumber, gchar *pnName );

/* address-dialog.c */
int getInfoByNumber( char *pnIncoming, char **ppnName, GdkPixbuf **ppsImage );
int SaveBook( void );
void addAddress( const gchar *name, const gchar *num );
GtkWidget *AddressBook( int nType );

/* audio.c */
void InitAudio( void );
void DeinitAudio( void );
void PlayRingtone( eCallType nType );
void StopRingtone( void  );

/* assistant.c */
int assistant( void );

/* callmonitor.c */
#include <filter.h>
void freeCallerList( void );
void getFilterInfo( struct sFilter *psFilter, struct sFilterInfo *psInfo );
void updateCallerList( const gchar *pnNumber, gchar *pnName );
gint callMonitorAddCaller( struct sCaller *psCaller, GtkListStore *psPrivStore, struct sFilter *psPrivFilter );

/* checklist.c */
void checklistReset( GtkWidget *psCheckList );
void checklistAddStage( GtkWidget *psCheckList, const gchar *pnName );
void checklistAddProgressbar( GtkWidget *psCheckList );
gboolean checklistCompleteStage( GtkWidget *psCheckList );
void checklistSetError( GtkWidget *psCheckList, const gchar *pnError );

/* configfile.c */
void prefsAddNone( struct sProfile *psProfile, const char *pnName );
void prefsSetString( struct sProfile *psProfile, const char *pnName, const char *pnValue );
void prefsSetPath( struct sProfile *psProfile, const char *pnName, const char *pnValue );
void prefsSetInt( struct sProfile *psProfile, const char *pnName, int nValue );
void prefsSetBool( struct sProfile *psProfile, const char *pnName, bool bValue );
const char *prefsGetPath( struct sProfile *psProfile, const char *pnName );
const char *prefsGetString( struct sProfile *psProfile, const char *pnName );
int prefsGetInt( struct sProfile *psProfile, const char *pnName );
gboolean prefsGetBool( struct sProfile *psProfile, const char *pnName );
void SavePreferences( struct sProfile *psProfile );
void SaveActivePreferences( void );
int InitPrefs( void );
char *xmlnode_to_formatted_str( xmlnode *psNode, int *pnLen );
int LoadPreferences( struct sProfile *psProfile );

/* common.c */
void commonParseFirmware( struct sProfile *psProfile );

/* csv.c */
extern GList *psCallerList;
extern GList *psPersonsList;
struct sPerson *AddPerson( GHashTable *psTable, gboolean bNew );
void freePersons( void );
void AddCall( const char *pnType, const char *pnDateTime, const char *pnName, const char *pnNumber, const char *pnLocalName, const char *pnLocalNumber, const char *pnDuration, const char *pnProfile );
void freeCallers( void );
int parseCsvData( char *pnData, char *pnProfileName );
int parseCsvFile( const char *pnName, char *pnProfileName );
struct sPerson *findPerson( const char *pnName );
void removePerson( const char *pnName );
struct sPerson *findPersonByNumber( const char *pnNumber );
void AddInfo( GHashTable *psTable, gint nKey, const gchar *pnValue );
int saveCallerList( void );

/* dialnumber.c */
void dialWindowSetStatus( GtkWidget *psDialog, gint nType );
void dialingDialog( const gchar *pnNumber, const gchar *pnPort );

/* callmonitor.c */
GtkWidget *CallMonitor( void );
gchar *getCallerList( struct sProfile *psProfile );

/* router.c */
int routerReconnect( struct sProfile *psProfile );
gchar *routerGetIp( struct sProfile *psProfile );
const gchar *routerGetName( struct sProfile *psProfile );
const gchar *routerGetVersion( struct sProfile *psProfile );

/* lastcall.c */
void LoadLastcalls( void );
void lastCallAddNumber( gint nType, gchar *pnNumber );

/* main.c */
gchar *getUiFile( const gchar *pnFile );
void setNameNumber( const gchar *pnName, const gchar *pnNumber );
void processFax( char *pnFile );
GdkPixbuf *getLogo( void );
void setTrayOpen( bool bSet );
GtkStatusIcon *getTrayIcon( void );

/* monitor.c */
gpointer monitor( gpointer data );
void InitMonitor( void );
void monitorClearConnection( void );
struct sConnection;
void *showNotification( struct sConnection *psConnection, eCallType nType, struct sPerson *psPerson, gchar *pnNumber, gchar *pnLocal );
gpointer updateCallList( gpointer data );

/* preferences.c */
const gchar *faxGetSenderMsn( struct sProfile *psProfile );
const char *faxGetSenderName( struct sProfile *psProfile );
const char *faxGetSenderNumber( struct sProfile *psProfile );
int faxGetController( struct sProfile *psProfile );
int faxGetBitrateIndex( struct sProfile *psProfile );
int faxGetBitrate( struct sProfile *psProfile );
int debugGetLevel( struct sProfile *psProfile );
int debugGetFaxLevel( struct sProfile *psProfile );
gboolean callMonitorGetIncoming( struct sProfile *psProfile, const gchar *pnNumber );
void callMonitorSetIncoming(  struct sProfile *psProfile, gboolean bSet );
gboolean callMonitorGetOutgoing( struct sProfile *psProfile, const gchar *pnNumber );
void callMonitorSetOutgoing( struct sProfile *psProfile,gboolean bSet );
gboolean faxGetAccept( struct sProfile *psProfile );
gboolean faxGetEcm( struct sProfile *psProfile );
void faxSetAcceptFax( struct sProfile *psProfile,gboolean bSet );
gboolean callMonitorGetReverseLookupIn( struct sProfile *psProfile );
void callMonitorSetReverseLookupIn( struct sProfile *psProfile,gboolean bSet );
gboolean callMonitorGetReverseLookupOut( struct sProfile *psProfile );
void callMonitorSetReverseLookupOut( struct sProfile *psProfile,gboolean bSet );
gboolean callMonitorGetLoadOnStartup( struct sProfile *psProfile );
void callMonitorSetLoadOnStartup( struct sProfile *psProfile, gboolean bSet );
const char *routerGetHost( struct sProfile *psProfile );
const char *routerGetPassword( struct sProfile *psProfile );
const char *routerGetDefaultFon( struct sProfile *psProfile );
int callMonitorGeDisplayTime( struct sProfile *psProfile );
const gchar *routerGetLineAccessCode( struct sProfile *psProfile );
const gchar *routerGetCountryCode( struct sProfile *psProfile );
const gchar *routerGetAreaCode( struct sProfile *psProfile );
gint callMonitorGetSoundType( struct sProfile *psProfile );
gboolean callMonitorGetLookup( struct sProfile *psProfile );
const char *routerGetFirmware( struct sProfile *psProfile );
const char *routerGetPortName( struct sProfile *psProfile, char *pnPort );
void updateProfiles( void );
const gchar *callMonitorGetOutgoingSound( struct sProfile *psProfile );
const gchar *callMonitorGetIncomingSound( struct sProfile *psProfile );
const gchar *faxGetReportDir( struct sProfile *psProfile );
const gchar *routerGetFonNumber( struct sProfile *psProfile, int nType );
int routerGetPorts( struct sProfile *psProfile );
int routerGetFonType( struct sProfile *psProfile, const gchar *pnNumber );
int routerGetFonFlags( struct sProfile *psProfile, int nType );
gint callMonitorGetPopupPosition( struct sProfile *psProfile );
int callMonitorGetDisplayTime( struct sProfile *psProfile );
gboolean callMonitorGetIncomingState( struct sProfile *psProfile, const gchar *pnNumber );
gboolean callMonitorGetOutgoingState( struct sProfile *psProfile, const gchar *pnNumber );
void routerSetStandardPorts( struct sProfile *psProfile );
void routerSetPassword( struct sProfile *psProfile, const gchar *pnPassword );
void routerSetHost( struct sProfile *psProfile, const char *pnHost );
void routerSetLineAccessCode( struct sProfile *psProfile, const gchar *pnCode );
void routerSetAreaCode( struct sProfile *psProfile, const gchar *pnCode );
void routerSetCountryCode( struct sProfile *psProfile, const gchar *pnCode );
const gchar *routerGetNationalPrefix( struct sProfile *psProfile );
void routerSetNationalPrefix( struct sProfile *psProfile, const gchar *pnPrefix );
const gchar *routerGetInternationalPrefix( struct sProfile *psProfile );
void rounerSetInternationalPrefix( struct sProfile *psProfile, const gchar *pnPrefix );
gboolean callMonitorGetLookupState( struct sProfile *psProfile );
void callMonitorGetSize( struct sProfile *psProfile, gint *pnWidth, gint *pnHeight );
void callMonitorSetSize( struct sProfile *psProfile, gint nWidth, gint nHeight );
void callMonitorSetPanedPosition( struct sProfile *psProfile, gint nPosition );
int callMonitorGetPanedPosition( struct sProfile *psProfile );
void routerAddComboBoxPorts( struct sProfile *psProfile, GtkWidget *psComboBox );
int routerGetPortType( struct sProfile *psProfile, const char *pnName );
gboolean faxGetReportToggle( struct sProfile *psProfile );
gboolean callMonitorGetMute( struct sProfile *psProfile );
void callMonitorSetMute( struct sProfile *psProfile, bool bSet );
int getActionFlags( struct sProfile *psProfile, const gchar *pnName );
void routerSetPorts( struct sProfile *psProfile, int nPorts );
void routerSetFirmware( struct sProfile *psProfile, const char *pnFirmware );
void routerClearNumbers( struct sProfile *psProfile );
void routerSetFonNumber( struct sProfile *psProfile, int nType, const gchar *pnMsn );
void routerSetPortName( struct sProfile *psProfile, int nType, char *pnName );
const gchar *pluginsGetLookup( struct sProfile *psProfile );
void pluginsSetLookup( struct sProfile *psProfile, const gchar *pnLookup );
int miscGetToolbarStyle( struct sProfile *psProfile );
gboolean callMonitorGetPlaySounds( struct sProfile *psProfile );
gboolean callMonitorGetType( struct sProfile *psProfile );
gboolean callMonitorGetDateTime( struct sProfile *psProfile );
gboolean callMonitorGetName( struct sProfile *psProfile );
gboolean callMonitorGetCompany( struct sProfile *psProfile );
gboolean callMonitorGetCity( struct sProfile *psProfile );
gboolean callMonitorGetNumber( struct sProfile *psProfile );
gboolean callMonitorGetLocalName( struct sProfile *psProfile );
gboolean callMonitorGetLocalNumber( struct sProfile *psProfile );
gboolean callMonitorGetDuration( struct sProfile *psProfile );
gboolean callMonitorGetReplaceOnLoad( struct sProfile *psProfile );
gint addressBookGetNumberFormat( struct sProfile *psProfile );
void addressBookSetNumberFormat( struct sProfile *psProfile, gint bSet );
const gchar *voiceBoxGetUser( struct sProfile *psProfile );
const gchar *voiceBoxGetPassword( struct sProfile *psProfile );
gboolean voiceBoxGetPulseaudio( struct sProfile *psProfile );
gboolean callMonitorGetSystemNotification( struct sProfile *psProfile );
gboolean callMonitorGetFfgtkNotification( struct sProfile *psProfile );
gint softphoneGetController( struct sProfile *psProfile );
const gchar *softphoneGetMsn( struct sProfile *psProfile );
const gchar *dtmfGetCaller( struct sProfile *psProfile );
const gchar *dtmfGetCalled( struct sProfile *psProfile );
const gchar *dtmfGetCode( struct sProfile *psProfile );
const gchar *dtmfGetExecute( struct sProfile *psProfile );

/* rl.c */
int ReverseLookup( gchar *pnNumber, gchar **ppnFirstName, gchar **ppnLastName, gchar **ppnStreet, gchar **ppnZipCode, gchar **ppnCity );
void LookupLoad( void );

/* print.c */
void PrintTreeView( GtkWidget *psWidget, gpointer pUserData );
struct sFaxStatus;
void CreateFaxReport( struct sFaxStatus *psStatus, char *pnReportDir );

/* profiles.c */
void ProfilesLoad( void );
void saveProfiles( void );
void removeProfile( const gchar *pnName );
const gchar *getProfilesDir( void );

/* router.c */
void DiversityLoad( void );
int routerGetActivePorts( struct sProfile *psProfile );
int routerFirmwareDetect( struct sProfile *psProfile );
int routerGetPhoneSettings( struct sProfile *psProfile );
int routerLogin( struct sProfile *psProfile );
int routerGetCallList( struct sProfile *psProfile );
gboolean routerHasDial( struct sProfile *psProfile );
int routerCallNumber( struct sProfile *psProfile, const gchar *pnNumber, const gchar *pnPort );
int routerLogout( struct sProfile *psProfile );
int routerDetection( struct sProfile *psProfile );
int routerInit( struct sProfile *psProfile );
void routerRemovePassword( struct sProfile *psProfile );
int routerCallListClear( struct sProfile *psProfile );
int routerHangup( struct sProfile *psProfile, const gchar *pnPort );
gint routerSetDiversity( struct sProfile *psProfile, gint nIndex, gboolean bSet );
void routerGetSpeed( struct sProfile *psProfile, gint *pnMaxUp, gint *pnMaxDown );
int routerLoadVoiceBox( struct sProfile *psProfile );
int routerLoadVoiceFile( struct sProfile *psProfile, const gchar *pnFile );
int routerDeleteVoiceFile( struct sProfile *psProfile, const gchar *pnFile );
int routerLoadFaxBox( struct sProfile *psProfile );
int routerLoadFaxFile( struct sProfile *psProfile, const gchar *pnFile );
int routerDeleteFaxFile( struct sProfile *psProfile, const gchar *pnFile );

/* server.c */
void PrintServerInit( void );

/* ssdp.c */
int ssdpDiscover( char **ppnHost, unsigned short *pnPort );

/* trayicon.c */
void TrayIconInit( void );
void recreateAppMenu( void );

/* misc.c */
gchar *getDirectory( gchar *pnType );
gchar *convertEntities( const gchar *pnString );
gchar *fullNumber( const char *pnNumber, gboolean bCountryCodePrefix );
gchar *normalizeNumber( const char *pnNumber );
gchar *getFileDate( const gchar *pnFile );
gchar *fixNumber( const gchar *pnNumber );
gchar *extractInformation( gchar **ppnLines, gchar *pnRegex );
gboolean extractInformationNamed( gchar *ppnLines, gchar **ppnFirstName, gchar **ppnLastName, gchar **ppnStreet, gchar **ppnZipCode, gchar **ppnCity, gchar *pnRegex );
GdkPixbuf *getScaledImage( struct sPerson *psPerson, float fScale );
void saveData( char *pnName, char *pnData, gint nLen );
void saveDebugData( char *pnName, char *pnData, gint nLen );
void ShowError( gboolean bThread, const gchar *pnErrorMessage, const gchar *pnText );
void addData( char *pnName, char *pnData );
void freeTypeIcons( void );
gchar *getCityByNumber( gchar *pnNumber );

void dialogNumberButtonClicked( GtkWidget *psWidget, gpointer pData );
gboolean dialNameMatchSelect( GtkEntryCompletion *psWidget, GtkTreeModel *psModel, GtkTreeIter *psIter, gpointer pUserData );
int preferences( void );
void dialNumberDialog( struct sPerson *psPerson, const gchar *pnNumber );
gchar *dialGetNumber( void );
GtkWidget *dialGetDialog( void );
const char * GFileMonitorEvent2Text( GFileMonitorEvent event);
void dialSetNumber( const gchar *pnNumber );

GtkWidget *checklistNew( void );

gchar *loadData( char *pnFileName, gint *pnSize );

gchar *replaceNumber( gchar *pnStr, gchar *pnLocal, gchar *pnNumber, gchar *pnName );
GList *getLookupList( const gchar *pnCountryCode );

GList *routerGetDiversity( void );
GList *getLastcalls( gint nType );
GdkPixbuf *getTypeIcon( int nType );

void setDialNameNumber( GtkWidget *psParent, const gchar *pnName, const gchar *pnNumber );

/* vcard.c */
void loadCardFile( char *pnFileName );
void writeCardFile( char *pnFileName );

/* xml.c */
void xmlnode_free( xmlnode *psNode );
void xmlnode_set_attrib( xmlnode *psNode, const char *pnAttr, const char *pnValue );
void xmlnode_insert_child( xmlnode *psParent, xmlnode *psChild );

/* sendsms.c */
void sendSmsDialog( void );

/* faxbox.c */
void faxBoxOpen( GtkTreeModel *psModel, GtkTreePath *psPath );
void faxBoxDelete( GtkTreeModel *psModel, GtkTreePath *psPath );

/* voicebox.c */
void voiceBoxPlay( GtkTreeModel *psModel, GtkTreePath *psPath );
void voiceBoxDelete( GtkTreeModel *psModel, GtkTreePath *psPath );

void softphoneHandleBluetooth( void );
int softphoneDial( gchar *pnNumber, gboolean bAnonymous );
void softphoneHangup();
void softphonePickup();
void softphoneSendDtmfCode( gint nNum );
void softphoneMute( gboolean bMute );
void softphoneHold( gboolean bHold );
void softphoneRecord( gboolean bRecord );
void softphoneFlush( void );
int faxophoneSendFax( gchar *pnFaxFile, gint nBitRate, gint nEcm, gint nController, const gchar *pnSender, gchar *pnTarget, const gchar *pnHeader, const gchar *pnIdent, gint nAnonymous, const gchar *pnReportDir );

gint PlayVoxFile( gchar *pnFile );

int InitBluetooth( void );
void InitFaxPhone( void );
void CloseFaxPhone( void );
gboolean has_file_extension( const char *pnFile, const char *pnExt );

void ffgtkLock( void );
void ffgtkUnlock( void );

char *g_strcasestr( const char *haystack, const char *needle );
gchar *convert_utf8( const char *pnString );

#if GTK_MAJOR_VERSION > 2
GtkWidget *gtk_combo_box_new_text( void );
void gtk_combo_box_append_text( GtkComboBox *psBox, const gchar *pnText );
gchar *gtk_combo_box_get_active_text( GtkComboBox *psBox );
#endif

#if GLIB_CHECK_VERSION(2, 31, 0)
#define CREATE_THREAD(name, func, data) g_thread_new(name, func, data)
#else
#define CREATE_THREAD(name, func, data) g_thread_create(func, data, false, NULL)
#endif

#endif
