/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 1991-2010 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Application
    compressibleInterDyMFoam

Description
    Solver for 2 compressible, isothermal immiscible fluids using a VOF
    (volume of fluid) phase-fraction based interface capturing approach,
    with optional mesh motion and mesh topology changes including adaptive
    re-meshing.

    The momentum and other fluid properties are of the "mixture" and a single
    momentum equation is solved.

    Turbulence modelling is generic, i.e. laminar, RAS or LES may be selected.

Usage
    - compressibleInterDyMFoam [OPTION]

    @param -case \<dir\> \n
    Specify the case directory

    @param -parallel \n
    Run the case in parallel

    @param -help \n
    Display short usage message

    @param -doc \n
    Display Doxygen documentation page

    @param -srcDoc \n
    Display source code

\*---------------------------------------------------------------------------*/

#include <finiteVolume/fvCFD.H>
#include <dynamicFvMesh/dynamicFvMesh.H>
#include <finiteVolume/MULES.H>
#include <OpenFOAM/subCycle.H>
#include <interfaceProperties/interfaceProperties.H>
#include <incompressibleTransportModels/twoPhaseMixture.H>
#include <incompressibleTurbulenceModel/turbulenceModel.H>

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

int main(int argc, char *argv[])
{
    #include <OpenFOAM/setRootCase.H>
    #include <OpenFOAM/createTime.H>
    #include <dynamicFvMesh/createDynamicFvMesh.H>
    #include <finiteVolume/readGravitationalAcceleration.H>
    #include "readControls.H"
    #include <finiteVolume/initContinuityErrs.H>
    #include "../createFields.H"
    #include "createPcorrTypes.H"
    #include <finiteVolume/CourantNo.H>
    #include <finiteVolume/setInitialDeltaT.H>

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
    Info<< "\nStarting time loop\n" << endl;

    while (runTime.run())
    {
        #include "readControls.H"
        #include <finiteVolume/CourantNo.H>

        // Make the fluxes absolute
        fvc::makeAbsolute(phi, U);

        #include <finiteVolume/setDeltaT.H>

        runTime++;

        Info<< "Time = " << runTime.timeName() << nl << endl;

        {
            // Store divU from the previous mesh for the correctPhi
            volScalarField divU = fvc::div(phi);

            scalar timeBeforeMeshUpdate = runTime.elapsedCpuTime();

            // Do any mesh changes
            mesh.update();

            if (mesh.changing())
            {
                Info<< "Execution time for mesh.update() = "
                    << runTime.elapsedCpuTime() - timeBeforeMeshUpdate
                    << " s" << endl;

                gh = g & mesh.C();
                ghf = g & mesh.Cf();
            }

            if (mesh.changing() && correctPhi)
            {
                #include "correctPhi.H"
            }
        }

        // Make the fluxes relative to the mesh motion
        fvc::makeRelative(phi, U);

        if (mesh.changing() && checkMeshCourantNo)
        {
            #include <dynamicFvMesh/meshCourantNo.H>
        }

        turbulence->correct();

        // --- Outer-corrector loop
        for (int oCorr=0; oCorr<nOuterCorr; oCorr++)
        {
            #include "alphaEqnsSubCycle.H"

            solve(fvm::ddt(rho) + fvc::div(rhoPhi));

            #include "../UEqn.H"

            // --- PISO loop
            for (int corr=0; corr<nCorr; corr++)
            {
                #include "pEqn.H"
            }
        }

        rho = alpha1*rho1 + alpha2*rho2;

        runTime.write();

        Info<< "ExecutionTime = "
            << runTime.elapsedCpuTime()
            << " s\n\n" << endl;
    }

    Info<< "End\n" << endl;

    return 0;
}


// ************************ vim: set sw=4 sts=4 et: ************************ //
